using System;
using System.Collections;
using UnityEditor;
using UnityEngine;
using UnityVolumeRendering;

// https://github.com/mlavik1/UnityVolumeRendering/blob/master/Assets/Editor/TransferFunctionEditorWindow.cs
public class TransferFunctionEditorWindow : EditorWindow
{
    private TransferFunction tf = null;

    private TransferFunctionEditor tfEditor = new TransferFunctionEditor();

    public static void ShowWindow(TransferFunction transferFunction)
    {
        var wnd = GetWindow<TransferFunctionEditorWindow>();
        wnd.tf = transferFunction;
        wnd.Show();
        wnd.SetInitialPosition();
    }
    
    private void SetInitialPosition()
    {
        var rect = this.position;
        rect.width = 800.0f;
        rect.height = 500.0f;
        this.position = rect;
    }

    private void OnEnable()
    {
        tfEditor.Initialise();
    }
    
     private void OnGUI()
        {
            wantsMouseEnterLeaveWindow = true;

            if (tf == null)
                return;

            Event currentEvent = new Event(Event.current);

            Color oldColour = GUI.color; // Used for setting GUI.color when drawing UI elements
            
            float contentWidth = Mathf.Min(this.position.width, (this.position.height - 100.0f) * 2.0f);
            float contentHeight = contentWidth * 0.5f;
            
            // Interaction area (slightly larger than the histogram rect)
            Rect outerRect = new Rect(0.0f, 0.0f, contentWidth, contentHeight);
            Rect tfEditorRect = new Rect(outerRect.x + 20.0f, outerRect.y + 20.0f, outerRect.width - 40.0f, outerRect.height - 50.0f);

            tfEditor.SetTransferFunction(tf);
            tfEditor.DrawOnGUI(tfEditorRect);

            // Draw horizontal zoom slider
            float horZoomMin = tfEditor.zoomRect.x;
            float horZoomMax = tfEditor.zoomRect.x + tfEditor.zoomRect.width;
            EditorGUI.MinMaxSlider(new Rect(tfEditorRect.x, tfEditorRect.y + tfEditorRect.height, tfEditorRect.width, 20.0f), ref horZoomMin, ref horZoomMax, 0.0f, 1.0f);
            tfEditor.zoomRect.x = horZoomMin;
            tfEditor.zoomRect.width = horZoomMax - horZoomMin;

            // Draw vertical zoom slider
            GUIUtility.RotateAroundPivot(270.0f, Vector2.zero);
            GUI.matrix = Matrix4x4.Translate(new Vector3(tfEditorRect.x + tfEditorRect.width, tfEditorRect.y + tfEditorRect.height, 0.0f)) * GUI.matrix;
            float vertZoomMin = tfEditor.zoomRect.y;
            float vertZoomMax = tfEditor.zoomRect.y + tfEditor.zoomRect.height;
            EditorGUI.MinMaxSlider(new Rect(0.0f, 0.0f, tfEditorRect.height, 20.0f), ref vertZoomMin, ref vertZoomMax, 0.0f, 1.0f);
            tfEditor.zoomRect.y = vertZoomMin;
            tfEditor.zoomRect.height = vertZoomMax - vertZoomMin;
            GUI.matrix = Matrix4x4.identity;

            // Save TF
            if(GUI.Button(new Rect(tfEditorRect.x, tfEditorRect.y + tfEditorRect.height + 20.0f, 70.0f, 30.0f), "Save"))
            {
                string filepath = EditorUtility.SaveFilePanel("Save transfer function", "", "default.tf", "tf");
                if(filepath != "")
                    tf.SaveToFile(filepath);
            }

            // Load TF
            if(GUI.Button(new Rect(tfEditorRect.x + 75.0f, tfEditorRect.y + tfEditorRect.height + 20.0f, 70.0f, 30.0f), "Load"))
            {
                string filepath = EditorUtility.OpenFilePanel("Save transfer function", "", "tf");
                if(filepath != "")
                {
                    tf.LoadFromFile(filepath);
                }
            }
             // Clear TF
            if(GUI.Button(new Rect(tfEditorRect.x + 150.0f, tfEditorRect.y + tfEditorRect.height + 20.0f, 70.0f, 30.0f), "Clear"))
            {
                tf.ClearControlPoints();
                tfEditor.ClearSelection();
            }

            Color? selectedColour = tfEditor.GetSelectedColor();
            if (selectedColour != null)
            {
                // Colour picker
                Color newColour = EditorGUI.ColorField(new Rect(tfEditorRect.x + 245, tfEditorRect.y + tfEditorRect.height + 20.0f, 100.0f, 40.0f), selectedColour.Value);
                if (selectedColour != newColour)
                    tfEditor.SetSelectedColour(newColour);

                // Remove colour
                if (GUI.Button(new Rect(tfEditorRect.x + 350.0f, tfEditorRect.y + tfEditorRect.height + 20.0f, 70.0f, 30.0f), "Remove"))
                    tfEditor.RemoveSelectedColor();
            }

            GUI.skin.label.wordWrap = false;    
            GUI.Label(new Rect(tfEditorRect.x, tfEditorRect.y + tfEditorRect.height + 55.0f, 720.0f, 50.0f), "Left click to select and move a control point.\nRight click to add a control point, and ctrl + right click to delete.");

            float tDataPos = (currentEvent.mousePosition.x - tfEditorRect.x) / tfEditorRect.width;
            if (tDataPos >= 0.0f && tDataPos <= 1.0f)
            {
                float dataValue = Mathf.Lerp(VolumeData.MIN_HOUNSFIELD, VolumeData.MAX_HOUNSFIELD, tDataPos);
                GUI.Label(new Rect(tfEditorRect.x, tfEditorRect.y + tfEditorRect.height + 100.0f, 150.0f, 50.0f), $"Data value: {dataValue}");
            }
            
            if (!tf.IsMonotonicIncreasing())
            {
                EditorGUILayout.HelpBox("This transfer function is not monotonic increasing. Therefore DDA is disabled and performance is significantly reduced.", MessageType.Warning);
            }

            GUI.color = oldColour;
        }

        private void OnSelectionChange()
        {
            var newTransferFunction = Selection.activeGameObject?.GetComponent<TransferFunction>();
            // If we selected another volume object than the one previously edited in this GUI
            if (newTransferFunction != null)
                this.Close();
        }

        public void OnInspectorUpdate()
        {
            Repaint();
        }
}

