using System;
using System.Collections.Generic;
using Unity.Mathematics;
using UnityEngine;
using UnityEngine.Rendering;
using Vector3 = UnityEngine.Vector3;

//[ExecuteAlways]
[ExecuteInEditMode]
public class VolumeObject : MonoBehaviour
{
    public static HashSet<VolumeObject> AllVolumes = new HashSet<VolumeObject>();//todo implement this
    
    public delegate void VolumeAdded(VolumeObject volumeObject);
    public static VolumeAdded OnVolumeAdded;

    public delegate void VolumeRemoved(VolumeObject volumeObject);
    public static VolumeRemoved OnVolumeRemoved;

    [SerializeField] private RenderType renderType = RenderType.MainDynamic;
    [SerializeField] private Vector3 albedo = Vector3.one * 0.9f;
    [SerializeField] private float phase = 0;
    [SerializeField] private float densityScale = 1.0f;

    [SerializeField] private int bounces = 100;
    //[SerializeField] private float emissionScale = 100.0f;

    [SerializeField] private MipSetting mipSetting;
    [SerializeField] private int rangeMipFixed = 0;

    [SerializeField] private VolumeData volume;

    public delegate void HasChanged();

    public HasChanged OnHasChanged;

    private TransferFunction transferFunction;

    public TransferFunction TransferFunction
    {
        get => transferFunction;
        set
        {
            OnHasChanged?.Invoke();
            transferFunction = value;
        }
    }

    public void Bind(ComputeShader shader, ShaderIndices shaderIndices)
    {
        volume.Bind(shader, shaderIndices.TraceKernel, shaderIndices);

        //shader.SetTexture(0, "vol_emission_indirection", volumeEmissionIndirection);
        //shader.SetTexture(0, "vol_emission_range", volumeEmissionRange);
        //shader.SetTexture(0, "vol_emission_atlas", volumeEmissionAtlas);

        shader.SetVector(shaderIndices.VolumeAlbedo, albedo);
        shader.SetFloat(shaderIndices.VolumePhaseG, phase);
        // shader.SetFloat(shaderIndices.VolumeDensityScale, densityScale);
        shader.SetFloat(shaderIndices.VolumeDensityScale, densityScale);
        //shader.SetFloat("vol_emission_scale", emissionScale);
        var (bbMin, bbMax) = GetVolAABB();
        shader.SetVector(shaderIndices.VolumeBBMin, bbMin);
        shader.SetVector(shaderIndices.VolumeBBMax, bbMax);
        // Debug.Log($"min = {bbMin}, max = {bbMax}");

        shader.SetInt(shaderIndices.Bounces, bounces);

        var transformMat = GetTransformMat();

        shader.SetMatrix(shaderIndices.VolumeDensityTransform, transformMat);

        shader.SetMatrix(shaderIndices.VolumeDensityInverseTransform, transformMat.inverse);

        if (transferFunction != null)
        {
            shader.EnableKeyword("USE_TRANSFERFUNC");
            transferFunction.Bind(shader, shaderIndices);
        }
        else
        {
            shader.DisableKeyword("USE_TRANSFERFUNC");
        }
        shader.DisableKeyword("RENDER_TYPE_MAIN");
        shader.DisableKeyword("RENDER_TYPE_DIRECT");
        shader.DisableKeyword("RENDER_TYPE_BOX_INTERSECT");
        switch (renderType)
        {
            case RenderType.MainDDA:
                shader.EnableKeyword("USE_DDA");
                shader.EnableKeyword("RENDER_TYPE_MAIN");
                break;
            case RenderType.MainNoDDA:
                shader.DisableKeyword("USE_DDA");
                shader.EnableKeyword("RENDER_TYPE_MAIN");
                break;
            case RenderType.MainDynamic:
                shader.EnableKeyword("RENDER_TYPE_MAIN");
                break;

            case RenderType.Direct:
                shader.EnableKeyword("RENDER_TYPE_DIRECT");
                break;

            case RenderType.BoxIntersection:
                shader.EnableKeyword("RENDER_TYPE_BOX_INTERSECT");
                break;
            default:
                break;
        }

        if (mipSetting == MipSetting.MipDynamic)
        {
            shader.DisableKeyword("MIP_FIXED");
        }
        else
        {
            shader.EnableKeyword("MIP_FIXED");
            shader.SetInt(shaderIndices.MipRangeFixed, rangeMipFixed);
        }
    }

    private (Vector3, Vector3) GetVolAABB() //todo there is a Bounds class for AABB
    {
        var transformMat = GetTransformMat();

        var extends = new Vector3(
            volume.dimensions.x,
            volume.dimensions.y,
            volume.dimensions.z);

        // extends /= 2f;
            
        var origin = Vector3.zero;
        var minBB = transformMat.MultiplyPoint3x4(origin);
        var maxBB = transformMat.MultiplyPoint3x4(origin + extends);
        //Debug.Log(minBB + " " + maxBB);
        return (minBB, maxBB);
    }

    private Matrix4x4 GetTransformMat()
    {
        // transform.localScale /= 100f;
        Matrix4x4 localToWorldMatrix = transform.localToWorldMatrix;
        // Debug.LogWarning($"{transform.lossyScale}");
        // transform.localScale *= 100f;
        return localToWorldMatrix;
    }


    private void OnEnable()
    {
        AllVolumes.Add(this);
        OnVolumeAdded?.Invoke(this);
        this.transform.localScale = volume.volumeScale;
        this.transform.localRotation = volume.volumeRotation;
    }

    private void OnDisable()
    {
        AllVolumes.Remove(this);
        OnVolumeRemoved?.Invoke(this);
    }

    private void Start()
    {
        // OnVolumeAdded?.Invoke(this);
        //ScaleToUnitCube();
        // this.transform.localScale = volume.volumeScale;
        // this.transform.localRotation = volume.volumeRotation;
        //ScaleToUnitCube();
        //Debug.Log(transform.TransformPoint(-Vector3.one * 0.5f));
    }

    // private void OnDestroy()
    // {
    //     OnVolumeRemoved?.Invoke(this);
    // }

    private void OnValidate()
    {
        OnHasChanged?.Invoke();
    }

    private void Update()
    {
        if (transform.hasChanged)
        {
            transform.hasChanged = false;
            OnHasChanged?.Invoke();
        }
    }

    private void ScaleToUnitCube()
    {
        var size = math.max(volume.dimensions.x, math.max(volume.dimensions.y, volume.dimensions.z));
        // var currentScale = transform.localScale;
        // transform.localScale = new Vector3(currentScale.x / size, currentScale.y / size, currentScale.z / size);
        transform.localScale /= size;
    }

    public VolumeData GetVolumeData()
    {
        return volume;
    }
}
