using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using UnityEditor;
using UnityEngine;

#if UVR_USE_SIMPLEITK
using System;
using System.Runtime.InteropServices;
using System.Threading.Tasks;
using itk.simple;
using Unity.Mathematics;
#endif

public class VolumeImporter
{
    public const string SourceFolder = "Assets/Resources/Volumes/";
    public const string TextureSourceFolder = SourceFolder + "Textures/";

    [MenuItem("Volume Rendering/Load dataset/Load DICOM")]
    private static void ImportDicom()
    {
        if (!SimpleITKManager.HasDownloadedBinaries())
        {
            Debug.LogError("SimpleITK has not been downloaded or is incomplete!");
            return;
        }
        if (!SimpleITKManager.IsSITKEnabled())
        {
            Debug.LogError("SimpleITK needs to be enabled!");
            return;
        }
#if UVR_USE_SIMPLEITK
        string dir = EditorUtility.OpenFolderPanel("Select a folder to load", "", "");
        if (string.IsNullOrEmpty(dir)) return; //Pressed cancel
        if (!Directory.Exists(dir))
        {
            Debug.LogError($"Failed to import. Selected folder does not exist.\n\"{dir}\"");
            return;
        }

        GameObject placeholder = new GameObject("Dispatcher");
        MainThreadDispatcher dispatcher = placeholder.AddComponent<MainThreadDispatcher>();
        Task.Run(() => ImportDicomInternal(dir, dispatcher));
#endif
    }

#if UVR_USE_SIMPLEITK
    private static async void ImportDicomInternal(string dir, MainThreadDispatcher dispatcher)
    {
        try
        {
            Debug.Log("Loading dataset.");

            IEnumerable<string> fileCandidates = Directory.EnumerateFiles(dir, "*.*", SearchOption.TopDirectoryOnly);

            if (!fileCandidates.Any())
            {
                Debug.LogError("No file candidates found!");
                return;
            }

            var seriesIDs = ImageSeriesReader.GetGDCMSeriesIDs(dir);

            foreach (var seriesID in seriesIDs)
            {
                Debug.Log("Loading dicom " + seriesID);
                var dicomNames = ImageSeriesReader.GetGDCMSeriesFileNames(dir, seriesID);
                var reader = new ImageSeriesReader();
                reader.SetFileNames(dicomNames);

                var image = reader.Execute();

                image = SimpleITK.DICOMOrient(image, "LSP");

                image = SimpleITK.Cast(image, PixelIDValueEnum.sitkFloat32);

                var dimensions = image.GetSize();

                var spacing = image.GetSpacing();
                var volumeScale = new Vector3( //todo I believe the scale of like the whole thing is used on a voxel? fix that
                    (float)spacing[0],
                    (float)spacing[1],
                    (float)spacing[2]
                );
                Debug.Log($"{volumeScale}");

                Debug.Log("Creating buffer.");

                var rawData = new Buf3D<float>(new uint3(dimensions[0], dimensions[1], dimensions[2]));

                var imgBuffer = image.GetBufferAsFloat();
                Marshal.Copy(imgBuffer, rawData.Data, 0, (int)rawData.GetSize());

                Debug.Log("Creating grid data.");

                Debug.Log("Dispatching to main render thread.");

                dispatcher.Dispatch(() =>
                {
                    //https://stackoverflow.com/questions/6018293/get-the-last-part-of-current-directory-name-in-c-sharp
                    string fullPath = Path.GetFullPath(dir).TrimEnd(Path.DirectorySeparatorChar);
                    string projectName = Path.GetFileName(fullPath);
                    
                    VolumeData volumeAsset = ScriptableObject.CreateInstance<VolumeData>();
                    volumeAsset.Init(rawData, projectName, volumeScale);

                    VolumeDataFormatWindow.ShowWindow(volumeAsset);
                });
            }

        }
        catch (Exception e)
        {
            dispatcher.Cancel();
            Debug.LogError(e);
        }
    }
#endif
    
    [MenuItem("Volume Rendering/Settings")]
    private static void ShowSettingsWindow()
    {
        ImportSettingsEditorWindow.ShowWindow();
    }
}
