﻿// Header guard
#ifndef TRACE_UTIL_HEADER_GUARD
#define TRACE_UTIL_HEADER_GUARD

// --------------------------------------------------------------
// constants and helper funcs

#define M_PI float(3.14159265358979323846)
#define INV_4_PI (1.f / (4 * M_PI))

float4 sanitize(const float4 x)
{
    return lerp(x, float4(0, 0, 0, 0), any(isinf(x)));
}

float sqr(const float x) { return x * x; }
float3 sqr(const float3 x) { return x * x; }

float luma(const float3 col) { return dot(col, float3(0.212671f, 0.715160f, 0.072169f)); }
//float luma(const float3 col) { return Luminance(col); }

float3 align(const float3 N, const float3 v) {
    // build tangent frame
    const float3 T = abs(N.x) > abs(N.y) ?
        float3(-N.z, 0, N.x) / sqrt(N.x * N.x + N.z * N.z) :
        float3(0, N.z, -N.y) / sqrt(N.y * N.y + N.z * N.z);
    const float3 B = cross(N, T);
    // tangent to world
    return normalize(v.x * T + v.y * B + v.z * N);
}

float power_heuristic(const float a, const float b) { return sqr(a) / (sqr(a) + sqr(b)); }

// --------------------------------------------------------------
// phase function helpers

float phase_henyey_greenstein(const float cos_t, const float g) {
    const float denominator = 1 + sqr(g) + 2 * g * cos_t;
    return INV_4_PI * (1 - sqr(g)) / (denominator * sqrt(denominator));
}

float3 sample_phase_henyey_greenstein(const float3 dir, const float g, const float2 phase_sample) {
    const float cos_t = abs(g) < 1e-4f ? 1.f - 2.f * phase_sample.x :
        (1 + sqr(g) - sqr((1 - sqr(g)) / (1 - g + 2 * g * phase_sample.x))) / (2 * g);
    const float sin_t = sqrt(max(0.f, 1.f - sqr(cos_t)));
    const float phi = 2.f * M_PI * phase_sample.y;
    return align(dir, float3(sin_t * cos(phi), sin_t * sin(phi), cos_t));
}

// --------------------------------------------------------------
// random number generation helpers

float rng(inout uint previous)
{
    previous = previous * 1664525u + 1013904223u;
    return float(previous & 0x00FFFFFFu) / float(0x01000000u);
}

float2 rng2(inout uint previous)
{
    return float2(rng(previous), rng(previous));
}

float3 rng3(inout uint previous)
{
    return float3(rng(previous), rng(previous), rng(previous));
}

float4 rng4(inout uint previous)
{
    return float4(rng(previous), rng(previous), rng(previous), rng(previous));
}

// tiny encryption algorithm (TEA) to calculate a seed per launch index and iteration
uint tea(const uint val0, const uint val1, const uint N)
{ 
    uint v0 = val0;
    uint v1 = val1;
    uint s0 = 0;
    for (uint n = 0; n < N; ++n) {
        s0 += 0x9e3779b9;
        v0 += ((v1 << 4) + 0xA341316C) ^ (v1 + s0) ^ ((v1 >> 5) + 0xC8013EA4);
        v1 += ((v0 << 4) + 0xAD90777D) ^ (v0 + s0) ^ ((v0 >> 5) + 0x7E95761E);
    }
    return v0;
}

#endif